/*
  Script Name: mono_CreateComps_v2
  Description: マーカーからコンポ作成＋情報追加（FPS選択機能実装版）
  
  Features:
  - Create Comps from Markers (Reverse Order)
  - Auto-add Cut Number & Timecode
  - GUI: Font Dropdown, Add Font Button, TC Start (0/1), Position Margins, FPS Selection
*/

(function(thisObj) {
    // --- デフォルト設定 ---
    var defaultSettings = {
        presetFonts: [
            "Meiryo-Bold",
            "Arial-BoldMT",
            "Helvetica-Bold",
            "KozGoPr6N-Medium",
            "KozGoPr6N-Bold",
            "YuGo-Bold",
            "MyriadPro-Bold"
        ],
        fontSize: 79,          
        referenceHeight: 1080, 
        // 余白のデフォルト値（%）
        defaultMarginX: 3.5, 
        defaultMarginY: 4.0,
        // FPSプリセット (59.94, 60を除外)
        fpsOptions: ["Source (Default)", "29.97", "24", "30", "Custom"]
    };

    // --- UI構築セクション ---
    var scriptName = "mono_CreateComps_v2";
    var myPanel = (thisObj instanceof Panel) ? thisObj : new Window("palette", scriptName, undefined, {resizeable:true});

    var res = "group{orientation:'column', alignment:['fill','top'], alignChildren:['fill','top'],\
        introText: StaticText{text:'マーカーからコンポ作成＆情報追加', alignment:['left','top']},\
        settingsPanel: Panel{text:'テキスト・レイアウト設定', orientation:'column', alignment:['fill','top'], alignChildren:['left','top'],\
            fontGroup: Group{orientation:'row', alignment:['fill','top'],\
                lbl: StaticText{text:'フォント:'},\
                dropdown: DropDownList{properties:{items:" + JSON.stringify(defaultSettings.presetFonts) + "}, alignment:['fill','center'], preferredSize:[150, -1]},\
                getBtn: Button{text:'フォントを追加', helpTip:'選択レイヤーのフォントをリストに追加'},\
            },\
            sizeGroup: Group{orientation:'row', alignment:['fill','top'],\
                lbl: StaticText{text:'基準サイズ(px):'},\
                input: EditText{text:'" + defaultSettings.fontSize + "', characters:5, properties:{enterKeyCausesNewLine:false}},\
                info: StaticText{text:'※1080p基準'},\
            },\
            marginGroup: Group{orientation:'row', alignment:['fill','top'],\
                lblX: StaticText{text:'横余白(%):'},\
                inputX: EditText{text:'" + defaultSettings.defaultMarginX + "', characters:4, properties:{enterKeyCausesNewLine:false}},\
                lblY: StaticText{text:'下余白(%):'},\
                inputY: EditText{text:'" + defaultSettings.defaultMarginY + "', characters:4, properties:{enterKeyCausesNewLine:false}},\
                help: StaticText{text:'(画面端からの距離)'},\
            },\
            fpsGroup: Group{orientation:'row', alignment:['fill','top'],\
                lbl: StaticText{text:'FPS:'},\
                dropdown: DropDownList{properties:{items:" + JSON.stringify(defaultSettings.fpsOptions) + "}, preferredSize:[120, -1]},\
                customInput: EditText{text:'', characters:6, enabled:false, helpTip:'Custom選択時に入力'},\
                unit: StaticText{text:'fps'},\
            },\
            tcGroup: Group{orientation:'row', alignment:['fill','top'],\
                lbl: StaticText{text:'TC開始:'},\
                rb1: RadioButton{text:'1 スタート (Default)', value: true},\
                rb0: RadioButton{text:'0 スタート'},\
            },\
        },\
        mainGroup: Group{orientation:'column', alignment:['fill','top'], alignChildren:['fill','top'],\
            addInfoChk: Checkbox{text:'カットNoとタイムコードを追加する', value: true},\
            runBtn: Button{text:'Create Comps', alignment:['fill','top']},\
        }\
    }";

    myPanel.grp = myPanel.add(res);

    // デフォルト値の設定
    myPanel.grp.settingsPanel.fontGroup.dropdown.selection = 0;
    
    // FPSドロップダウンのデフォルト選択 (Source)
    var fpsDd = myPanel.grp.settingsPanel.fpsGroup.dropdown;
    fpsDd.selection = 0;

    // --- UIイベント処理: FPSドロップダウン変更時 ---
    fpsDd.onChange = function() {
        var customInput = myPanel.grp.settingsPanel.fpsGroup.customInput;
        // 最後の項目(Custom)が選ばれたときだけ入力を有効化
        if (this.selection.index === (this.items.length - 1)) {
            customInput.enabled = true;
            if (customInput.text === "") customInput.text = "23.976"; 
        } else {
            customInput.enabled = false;
            customInput.text = "";
        }
    };

    // --- UIイベント処理: フォント追加ボタン ---
    myPanel.grp.settingsPanel.fontGroup.getBtn.onClick = function() {
        var activeItem = app.project.activeItem;
        if (!activeItem || !(activeItem instanceof CompItem)) {
            alert("コンポジションを開き、テキストレイヤーを選択してください。");
            return;
        }
        var sel = activeItem.selectedLayers;
        if (sel.length === 0 || !(sel[0] instanceof TextLayer)) {
            alert("テキストレイヤーを1つ選択してください。");
            return;
        }
        
        var textProp = sel[0].property("Source Text");
        var textDoc = textProp.value;
        var fontName = textDoc.font;

        var dropdown = myPanel.grp.settingsPanel.fontGroup.dropdown;
        
        var foundIndex = -1;
        for (var i = 0; i < dropdown.items.length; i++) {
            if (dropdown.items[i].text === fontName) {
                foundIndex = i;
                break;
            }
        }

        if (foundIndex !== -1) {
            dropdown.selection = foundIndex;
        } else {
            var item = dropdown.add("item", fontName);
            dropdown.selection = dropdown.items.length - 1;
        }
    };

    // --- メイン処理 ---
    myPanel.grp.mainGroup.runBtn.onClick = function() {
        app.beginUndoGroup(scriptName + " 実行");

        try {
            var activeComp = app.project.activeItem;
            if (!(activeComp instanceof CompItem)) {
                alert("コンポジションがアクティブではありません。");
                return;
            }

            var selectedFontItem = myPanel.grp.settingsPanel.fontGroup.dropdown.selection;
            if (!selectedFontItem) {
                alert("フォントが選択されていません。");
                return;
            }

            // --- FPSの決定ロジック ---
            var fpsSelection = myPanel.grp.settingsPanel.fpsGroup.dropdown.selection;
            var targetFPS = activeComp.frameRate; // デフォルトはソースと同じ

            if (fpsSelection.index === 0) {
                // Source (Default) なのでそのまま
                targetFPS = activeComp.frameRate;
            } else if (fpsSelection.index === (myPanel.grp.settingsPanel.fpsGroup.dropdown.items.length - 1)) {
                // Custom (リストの最後が選択されている場合)
                var customVal = parseFloat(myPanel.grp.settingsPanel.fpsGroup.customInput.text);
                if (isNaN(customVal) || customVal <= 0) {
                    alert("カスタムFPSには有効な数値を入力してください。");
                    return;
                }
                targetFPS = customVal;
            } else {
                // プリセット (文字列から数値へ変換)
                targetFPS = parseFloat(fpsSelection.text);
            }

            // UIから数値を取得
            var fontSizeVal = parseFloat(myPanel.grp.settingsPanel.sizeGroup.input.text);
            var marginXVal = parseFloat(myPanel.grp.settingsPanel.marginGroup.inputX.text);
            var marginYVal = parseFloat(myPanel.grp.settingsPanel.marginGroup.inputY.text);

            if (isNaN(fontSizeVal) || isNaN(marginXVal) || isNaN(marginYVal)) {
                alert("サイズや余白には半角数値を入力してください。");
                return;
            }

            // ラジオボタンの状態を確認
            var isZeroStart = myPanel.grp.settingsPanel.tcGroup.rb0.value;

            var currentSettings = {
                font: selectedFontItem.text,
                fontSize: fontSizeVal,
                referenceHeight: defaultSettings.referenceHeight,
                margins: { 
                    x: marginXVal / 100, 
                    y: marginYVal / 100 
                },
                isZeroStart: isZeroStart, 
                fontColor: [1, 1, 1],
                applyStroke: true,
                strokeColor: [0, 0, 0],
                strokeWidth: 4,
                strokeOverFill: true
            };

            var selectedLayers = activeComp.selectedLayers;
            if (selectedLayers.length !== 1) {
                alert("マーカーがあるレイヤーを1つだけ選択してください。");
                return;
            }

            var sourceLayer = selectedLayers[0];
            var markerProp = sourceLayer.property("Marker");

            if (markerProp.numKeys === 0) {
                alert("選択したレイヤーにマーカーがありません。");
                return;
            }

            var shouldAddInfo = myPanel.grp.mainGroup.addInfoChk.value;
            var createdCount = 0;

            for (var i = markerProp.numKeys; i >= 1; i--) {
                var currentMarker = markerProp.keyValue(i);
                var markerTime = markerProp.keyTime(i);
                
                var compName = currentMarker.comment;
                if (compName.replace(/\s/g, "") === "") { 
                    compName = sourceLayer.name + "_Marker_" + i;
                }

                var duration;
                if (i < markerProp.numKeys) {
                    var nextMarkerTime = markerProp.keyTime(i + 1);
                    duration = nextMarkerTime - markerTime;
                } else {
                    duration = sourceLayer.outPoint - markerTime;
                }
                
                if (duration <= 0) continue; 

                // コンポジション作成
                var newComp = app.project.items.addComp(
                    compName,
                    activeComp.width,
                    activeComp.height,
                    activeComp.pixelAspect,
                    duration,
                    targetFPS
                );

                var newCompLayer = activeComp.layers.add(newComp);
                newCompLayer.startTime = markerTime;
                
                if (shouldAddInfo) {
                    addCutInfoToComp(newComp, currentSettings);
                }

                createdCount++;
            }

        } catch(err) {
            alert("エラーが発生しました: " + err.toString());
        } finally {
            app.endUndoGroup();
        }
    };

    // --- サブ関数 ---
    function addCutInfoToComp(targetComp, s) {
        var w = targetComp.width;
        var h = targetComp.height;

        var scaleFactor = h / s.referenceHeight;
        var calcFontSize = s.fontSize * scaleFactor;
        var calcStrokeWidth = s.strokeWidth * scaleFactor;

        // CUT_NUM
        var cutNumLayer = targetComp.layers.addText("");
        cutNumLayer.name = "CUT_NUM";
        var cutNumDoc = cutNumLayer.property("Source Text").value;
        
        applyTextStyle(cutNumDoc, s, calcFontSize, calcStrokeWidth);
        cutNumDoc.justification = ParagraphJustification.LEFT_JUSTIFY;
        
        cutNumLayer.property("Source Text").setValue(cutNumDoc);
        cutNumLayer.property("Source Text").expression = 'thisComp.name;';
        
        // 位置計算
        cutNumLayer.property("Position").setValue([w * s.margins.x, h * (1 - s.margins.y)]);

        // TIMECODE
        var timecodeLayer = targetComp.layers.addText("");
        timecodeLayer.name = "TIMECODE";
        var timecodeDoc = timecodeLayer.property("Source Text").value;

        applyTextStyle(timecodeDoc, s, calcFontSize, calcStrokeWidth);
        timecodeDoc.justification = ParagraphJustification.RIGHT_JUSTIFY;

        timecodeLayer.property("Source Text").setValue(timecodeDoc);
        
        var addFrame = s.isZeroStart ? 0 : 1;
        timecodeLayer.property("Source Text").expression =
            'f = timeToFrames(time, 1/thisComp.frameDuration, true);\n' +
            '("000000" + (f + ' + addFrame + ')).slice(-6);';

        // 位置計算
        timecodeLayer.property("Position").setValue([w * (1 - s.margins.x), h * (1 - s.margins.y)]);

        var layers = [cutNumLayer, timecodeLayer];
        for (var k = 0; k < layers.length; k++) {
            layers[k].inPoint = 0;
            layers[k].outPoint = targetComp.duration;
            layers[k].shy = true; 
            layers[k].locked = true; 
        }
    }

    function applyTextStyle(textDoc, s, fSize, sWidth) {
        textDoc.font = s.font;
        textDoc.fontSize = fSize;
        textDoc.fillColor = s.fontColor;
        textDoc.tracking = 0;
        if (s.applyStroke) {
            textDoc.applyStroke = true;
            textDoc.strokeColor = s.strokeColor;
            textDoc.strokeWidth = sWidth;
            textDoc.strokeOverFill = s.strokeOverFill;
        } else {
            textDoc.applyStroke = false;
        }
    }

    myPanel.layout.layout(true);
    if (myPanel instanceof Window) {
        myPanel.center();
        myPanel.show();
    }

})(this);